<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["status" => false, "message" => "Only POST requests are allowed."]);
    exit;
}

require_once 'db.php';
require_once 'function.php';

try {
    // ✅ Validate required fields
    $requiredFields = ['user_id', 'name', 'dob', 'oppucation_type', 'oppucation_content'];
    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            throw new Exception("$field is required.");
        }
    }

    // ✅ Sanitize input
    $user_id = htmlspecialchars(trim($_POST['user_id']));
    $name = htmlspecialchars(trim($_POST['name']));
    $userName = htmlspecialchars(trim($_POST['userName'] ?? ''));
    $dob = htmlspecialchars(trim($_POST['dob']));
    $gender = htmlspecialchars(trim($_POST['gender'] ?? ''));
    $oppucation_type = htmlspecialchars(trim($_POST['oppucation_type']));
    $oppucation_content = htmlspecialchars(trim($_POST['oppucation_content']));
    $now = date('Y-m-d H:i:s');
    $profile_url = '';

    // ✅ Handle profile image upload
    if (isset($_FILES['profile']) && $_FILES['profile']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/userprofile/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }

        $fileTmpPath = $_FILES['profile']['tmp_name'];
        $fileName = basename($_FILES['profile']['name']);
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        $allowedExts = ['jpg', 'jpeg', 'png', 'webp'];

        if (!in_array($fileExt, $allowedExts)) {
            throw new Exception("Invalid image type. Allowed: jpg, jpeg, png, webp");
        }

        $newFileName = uniqid('profile_', true) . '.' . $fileExt;
        $destPath = $uploadDir . $newFileName;

        if (!move_uploaded_file($fileTmpPath, $destPath)) {
            throw new Exception("Failed to upload profile image.");
        }

        $profile_url = 'userprofile/' . $newFileName;

        // Update profile_url
        $stmt = $pdo->prepare("UPDATE users SET profile_url = :profile_url WHERE user_id = :user_id");
        $stmt->bindParam(':profile_url', $profile_url);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();
    }

    // ✅ Update users table
    $stmt = $pdo->prepare("UPDATE users SET name = :name, userName = :userName WHERE user_id = :user_id");
    $stmt->bindParam(':name', $name);
    $stmt->bindParam(':userName', $userName);
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();

    // ✅ Update user_kyc table
    $stmt = $pdo->prepare("
        UPDATE user_kyc 
        SET name = :name, dob = :dob, gender = :gender, 
            oppucation_type = :oppucation_type, 
            oppucation_content = :oppucation_content, 
            updated_at = :updated_at
        WHERE user_id = :user_id
    ");
    $stmt->bindParam(':name', $name);
    $stmt->bindParam(':dob', $dob);
    $stmt->bindParam(':gender', $gender);
    $stmt->bindParam(':oppucation_type', $oppucation_type);
    $stmt->bindParam(':oppucation_content', $oppucation_content);
    $stmt->bindParam(':updated_at', $now);
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();

    // ✅ Fetch updated data
    $stmt = $pdo->prepare("SELECT * FROM user_kyc WHERE user_id = :user_id");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    $kycData = $stmt->fetch(PDO::FETCH_ASSOC);

    echo json_encode([
        "status" => "success",
        "message" => "Profile updated successfully.",
        "userName" => $userName,
        "profile_url" => $profile_url,
        "data" => $kycData
    ]);

} catch (Exception $e) {
    // ✅ Handle errors gracefully
    http_response_code(400);
    echo json_encode([
        "status" => false,
        "message" => $e->getMessage()
    ]);
}
